/**
 * API Route: Single List
 * PUT /api/notes/lists/[id] - עדכון רשימה
 * DELETE /api/notes/lists/[id] - מחיקת רשימה
 */

import { NextRequest, NextResponse } from 'next/server';
import { getNotesDatabase } from '@/src/lib/notes-db';

export async function PUT(
    request: NextRequest,
    { params }: { params: Promise<{ id: string }> }
) {
    try {
        const { id } = await params;
        const body = await request.json();
        const db = await getNotesDatabase();

        const updates: string[] = [];
        const values: any[] = [];

        if (body.name !== undefined) {
            updates.push('name = ?');
            values.push(body.name);
        }
        if (body.color !== undefined) {
            updates.push('color = ?');
            values.push(body.color);
        }
        if (body.icon !== undefined) {
            updates.push('icon = ?');
            values.push(body.icon);
        }
        if (body.sortOrder !== undefined) {
            updates.push('sort_order = ?');
            values.push(body.sortOrder);
        }

        updates.push('updated_at = ?');
        values.push(new Date().toISOString());
        values.push(id);

        await db.run(
            `UPDATE lists SET ${updates.join(', ')} WHERE id = ?`,
            values
        );

        const list = await db.get('SELECT * FROM lists WHERE id = ?', [id]);

        return NextResponse.json({ list });
    } catch (error: any) {
        console.error('Error updating list:', error);
        return NextResponse.json(
            { error: error.message },
            { status: 500 }
        );
    }
}

export async function DELETE(
    request: NextRequest,
    { params }: { params: Promise<{ id: string }> }
) {
    try {
        const { id } = await params;
        const { searchParams } = new URL(request.url);
        const moveNotesToListId = searchParams.get('moveNotesToListId');

        const db = await getNotesDatabase();

        // If moveNotesToListId is provided, move notes to that list
        if (moveNotesToListId) {
            await db.run(
                'UPDATE notes SET list_id = ? WHERE list_id = ?',
                [moveNotesToListId, id]
            );
        } else {
            // Otherwise, set notes' list_id to null (inbox)
            await db.run(
                'UPDATE notes SET list_id = NULL WHERE list_id = ?',
                [id]
            );
        }

        await db.run('DELETE FROM lists WHERE id = ?', [id]);

        return NextResponse.json({ success: true });
    } catch (error: any) {
        console.error('Error deleting list:', error);
        return NextResponse.json(
            { error: error.message },
            { status: 500 }
        );
    }
}

